"""Groceries tasks."""


import numpy as np

from bigym.bigym_env import BiGymEnv
from bigym.const import HandSide
from bigym.envs.props.cabintets import BaseCabinet, WallCabinet, OpenShelf
from bigym.envs.props.items import GroceryCategory, GroceryItem
from bigym.utils.env_utils import get_random_points_on_plane


class GroceriesStoreLower(BiGymEnv):
    """Put groceries to lower cabinets tasks."""

    _CUPBOARD_1_POS = np.array([1.1, 0, 0])
    _CUPBOARD_2_POS = np.array([1.1, -0.6, 0])
    _CUPBOARD_ROT = np.array([0, 0, -np.pi / 2])

    _ITEM_CATEGORIES = [
        GroceryCategory.WINE,
        GroceryCategory.SOAP,
        GroceryCategory.BEER,
        GroceryCategory.CEREALS,
        GroceryCategory.KETCHUP,
        GroceryCategory.MUSTARD,
        GroceryCategory.SODA,
    ]

    _ITEMS_PER_CATEGORY = 1
    _ITEMS_COUNT = 4

    _ITEMS_POS = np.array([0.7, -0.3, 0.9])
    _ITEMS_STEP = 0.2
    _ITEMS_POS_EXTENTS = np.array([0.1, 0.5])
    _ITEMS_POS_BOUNDS = np.array([0.1, 0.05, 0])
    _ITEMS_ROT_BOUNDS = np.deg2rad([0, 0, 180])

    def _initialize_env(self):
        self.cupboard_base_1 = BaseCabinet(
            self._mojo,
            shelf_enable=True,
            door_right_enable=True,
        )
        self.cupboard_base_2 = BaseCabinet(
            self._mojo,
            shelf_enable=False,
            door_left_enable=True,
        )

        self.cupboard_base_1.body.set_position(self._CUPBOARD_1_POS)
        self.cupboard_base_1.body.set_euler(self._CUPBOARD_ROT)
        self.cupboard_base_2.body.set_position(self._CUPBOARD_2_POS)
        self.cupboard_base_2.body.set_euler(self._CUPBOARD_ROT)

        self.items: list[GroceryItem] = []
        self.selected_items: list[GroceryItem] = []
        for category in self._ITEM_CATEGORIES:
            self.items.extend(
                [
                    GroceryItem(self._mojo, category)
                    for _ in range(self._ITEMS_PER_CATEGORY)
                ]
            )

    def _on_reset(self):
        for item in self.items:
            item.disable()

        self.selected_items = np.random.choice(
            np.array(self.items), size=self._ITEMS_COUNT, replace=False
        )
        points = get_random_points_on_plane(
            len(self.selected_items),
            self._ITEMS_POS,
            self._ITEMS_POS_EXTENTS,
            self._ITEMS_STEP,
        )
        for item, point in zip(self.selected_items, points):
            item.enable()
            item.set_pose(
                point,
                position_bounds=self._ITEMS_POS_BOUNDS,
                rotation_bounds=self._ITEMS_ROT_BOUNDS,
            )

    def _success(self) -> bool:
        for item in self.selected_items:
            for side in HandSide:
                if self.robot.is_gripper_holding_object(item, side):
                    return False
            if not (
                item.is_colliding(self.cupboard_base_1.shelf)
                or item.is_colliding(self.cupboard_base_1.shelf_bottom)
                or item.is_colliding(self.cupboard_base_2.shelf_bottom)
            ):
                return False
        return True


class GroceriesStoreUpper(GroceriesStoreLower):
    """Put groceries to upper cabinets tasks."""

    def _initialize_env(self):
        self.cupboard_wall = WallCabinet(self._mojo, glass_doors_enable=True)
        self.shelf = OpenShelf(self._mojo)

        self.cupboard_wall.body.set_position(self._CUPBOARD_1_POS)
        self.cupboard_wall.body.set_euler(self._CUPBOARD_ROT)
        self.shelf.body.set_position(self._CUPBOARD_2_POS)
        self.shelf.body.set_euler(self._CUPBOARD_ROT)

        super()._initialize_env()

    def _success(self) -> bool:
        for item in self.selected_items:
            for side in HandSide:
                if self.robot.is_gripper_holding_object(item, side):
                    return False
            if not (
                item.is_colliding(self.cupboard_wall.shelf)
                or item.is_colliding(self.cupboard_wall.shelf_bottom)
                or item.is_colliding(self.shelf.shelf)
            ):
                return False
        return True
